#include "slave.h"

#include "EasyCAT.h"

#include <QFile>
#include <QTextStream>
#include <QRegularExpression>
#include <QTimer>

#include "../thread.h"

EtherCATSlave* startEtherCATSlaveInNewThread (int intervalMs) {
	QThread *thread = new QThread();
	thread->start();
	return executeInThreadAndWait<EtherCATSlave*>(thread, [intervalMs]{
		EtherCATSlave *slave = new EtherCATSlave();
		slave->loop(intervalMs);
		return slave;
	});
}

void stopEtherCATSlave (EtherCATSlave *slave) {
	executeOnObject<EtherCATSlave>(slave, (std::function<void(EtherCATSlave*)>) [](EtherCATSlave* slave) {
		auto thread = slave->thread();
		delete slave;
		thread->quit();
	});
}

static QString getRaspberryPiModel () {
	QFile fileCpu("/proc/cpuinfo");
	if (!fileCpu.open(QIODevice::ReadOnly)) {
		qFatal("Failed to open /proc/cpuinfo");
	}
	
	QRegularExpression reModel("^Model\t*: Raspberry Pi ([0-9]) .*$");
	
	QTextStream stream(&fileCpu);
	while (true) {
		QString line = stream.readLine();
		if (line.isNull()) {
			qFatal("Raspberry Pi Model detection failed");
		}
		QRegularExpressionMatch match = reModel.match(line);
		if (match.hasMatch()) {
			return match.captured(1);
		}
	}
}

EtherCATSlave::EtherCATSlave () : c(new EasyCAT()) {
	// Connect
	connect(this, &EtherCATSlave::threadSafeRegisterVar, this, &EtherCATSlave::registerVar);
	connect(this, &EtherCATSlave::threadSafeSetValue, this, &EtherCATSlave::setValue);
	
	// EasyCAT.h has input and output reversed!
	bufferIn = (char*) c->BufferOut.Byte;
	bufferOut = (char*) c->BufferIn.Byte;
	sizeInput = sizeof(c->BufferOut);
	sizeOutput = sizeof(c->BufferIn);
	
	// Keep track of variables
	vars.resize(sizeInput);
	
	// Set clock divider
	QString model = getRaspberryPiModel();
	uint16_t bcm2835_spi_clock_divider;
	if (model == "3")
		bcm2835_spi_clock_divider = BCM2835_SPI_CLOCK_DIVIDER_16;
	else if (model == "4")
		bcm2835_spi_clock_divider = BCM2835_SPI_CLOCK_DIVIDER_32;
	else
		qFatal("Unknown Raspberry Pi model");
	
	// Initialize
	if (!c->Init(bcm2835_spi_clock_divider)) {
		qFatal("EasyCAT initialization failed");
	}
}

EtherCATSlave::~EtherCATSlave () {
	delete c;
}

void EtherCATSlave::registerVar (int byteIndex, EtherCATInput *var, QByteArray value) {
	int size = var->size();
	for (int i=0; i<size; i++) {
		auto& p = vars[byteIndex + i];
		Q_ASSERT(p == nullptr);
		p = var;
		bufferIn[byteIndex + i] = value[i];
	}
}

void EtherCATSlave::setValue (int byteIndex, QByteArray value) {
	for (int i=0; i<value.size(); i++) {
		bufferOut[byteIndex + i] = value[i];
	}
}

void EtherCATSlave::process () {
	// Save input
	QByteArray oldInput = QByteArray(bufferIn, sizeInput);
	
	// Execute EasyCAT
	c->MainTask();
	
	// Update vars
	EtherCATInput *lastVar = nullptr;
	for (int i=0; i<sizeInput; i++) {
		if (bufferIn[i] != oldInput[i] && vars[i] != nullptr && vars[i] != lastVar) {
			lastVar = vars[i];
			
			// Find variable start position
			int start = i;
			while (start != -1 && vars[start] == lastVar) start--;
			start++;
			
			// Update
			lastVar->threadSafeUpdate(QByteArray(bufferIn + start, lastVar->size()));
		}
	}
}

void EtherCATSlave::loop (int intervalMs) {
	QTimer *timer = new QTimer(this);
	timer->setInterval(intervalMs);
	timer->start();
	connect(timer, &QTimer::timeout, this, &EtherCATSlave::process);
}
